/****************************************************************************
 ****************************************************************************
 *
 * example.c
 *
 ****************************************************************************
 ****************************************************************************/





#include <stdio.h>
#include <string.h>

#include "isat3.h"



/****************************************************************************
 * howto_get_the_interval_of_a_variable
 ****************************************************************************/
static void
howto_get_the_interval_of_a_variable(
	struct isat3			*is3,
	struct isat3_node		*variable,
	i3_tframe_t			tframe)

	{
	if (variable != NULL)
		{
		printf("tframe %d: %s %s%1.40f, %1.40f%s\n",
			tframe,
			isat3_node_get_variable_name(is3, variable),
			isat3_is_lower_bound_strict(is3, variable, tframe) ? "(" : "[",
			isat3_get_lower_bound(is3, variable, tframe),
			isat3_get_upper_bound(is3, variable, tframe),
			isat3_is_upper_bound_strict(is3, variable, tframe) ? ")" : "]");
		}
	}



/****************************************************************************
 * howto_examine_the_result
 ****************************************************************************/
static void
howto_examine_the_result(
	struct isat3			*is3,
	struct isat3_node		*variable1,
	struct isat3_node		*variable2,
	struct isat3_node		*variable3,
	struct isat3_node		*variable4,
	struct isat3_node		*variable5,
	struct isat3_node		*variable6,
	i3_type_t			result)

	{
	i3_tframe_t			tframe;
	i3_tframe_t			t;

	tframe = isat3_get_tframe(is3);
	printf("%s (in tframe %d)\n", isat3_get_result_string(result), tframe);
	if ((isat3_result_contains_possible_solution(result)) ||
		(isat3_result_contains_solution(result)))
		{
		for (t = 0; t <= tframe; t++)
			{
			howto_get_the_interval_of_a_variable(is3, variable1, t);
			howto_get_the_interval_of_a_variable(is3, variable2, t);
			howto_get_the_interval_of_a_variable(is3, variable3, t);
			howto_get_the_interval_of_a_variable(is3, variable4, t);
			howto_get_the_interval_of_a_variable(is3, variable5, t);
			howto_get_the_interval_of_a_variable(is3, variable6, t);
			}
		}
	printf("\n");
	}



/****************************************************************************
 * howto_build_an_expression_with_node_operations
 ****************************************************************************/
static struct isat3_node *
howto_build_an_expression_with_node_operations(
	struct isat3			*is3,
	struct isat3_node		*variable_w,
	struct isat3_node		*variable_x,
	struct isat3_node		*variable_y,
	struct isat3_node		*variable_z)

	{
	struct isat3_node		*constant_2;
	struct isat3_node		*mult;
	struct isat3_node		*plus1;
	struct isat3_node		*plus2;
	struct isat3_node		*equal;
	struct isat3_node		*minus;
	struct isat3_node		*less;
	struct isat3_node		*and;

	/*
	 * build up ((w = x + 2*y + z) and (w < -x)) using node operations.
	 * there are unary and binary operations
	 */

	constant_2 = isat3_node_create_constant_integer(is3, 2);
	mult = isat3_node_create_binary_operation(
		is3,
		ISAT3_NODE_BOP_MULT,
		constant_2,
		variable_y);
	plus1 = isat3_node_create_binary_operation(
		is3,
		ISAT3_NODE_BOP_ADD,
		variable_x,
		mult);
	plus2 = isat3_node_create_binary_operation(
		is3,
		ISAT3_NODE_BOP_ADD,
		plus1,
		variable_z);
	equal = isat3_node_create_binary_operation(
		is3,
		ISAT3_NODE_BOP_EQUAL,
		variable_w,
		plus2);
	minus = isat3_node_create_unary_operation(
		is3,
		ISAT3_NODE_UOP_MINUS,
		variable_x);
	less = isat3_node_create_binary_operation(
		is3,
		ISAT3_NODE_BOP_LESS,
		variable_w,
		minus);
	and = isat3_node_create_binary_operation(
		is3,
		ISAT3_NODE_BOP_AND,
		equal,
		less);

	/*
	 * destroy isat3_nodes only needed during constructing the wanted
	 * expression. destroying these nodes will decrement their
	 * reference counters. only if these nodes are not used by any
	 * other nodes they will be really destroyed. so in this case
	 * the nodes still exist, but and will automatically disappear if
	 * the parent node is destroyed
	 */

	isat3_node_destroy(is3, less);
	isat3_node_destroy(is3, minus);
	isat3_node_destroy(is3, equal);
	isat3_node_destroy(is3, plus2);
	isat3_node_destroy(is3, plus1);
	isat3_node_destroy(is3, mult);
	isat3_node_destroy(is3, constant_2);

	/* return the isat3_node representing the whole expression */

	return (and);
	}



/****************************************************************************
 * howto_build_an_expression_from_a_string
 ****************************************************************************/
static struct isat3_node *
howto_build_an_expression_from_a_string(
	struct isat3			*is3)

	{
	struct isat3_node		*expr;

	/*
	 * build up ((w^2 = x^3 + y^4 + z^5) and (w != 0) and (x != 0) and
	 * (y != 0) and (z != 0)) from a string
	 *
	 * - every variable occuring in the string has to be declared before
	 * - the string has to contain at least one clause which has to be
	 *   delimited with a ';'
	 * - multiple clauses are automatically conjuncted (this means ';'
	 *   is just another form of 'and')
	 */

	expr = isat3_node_create_from_string(is3,
		"(w^2 = x^3 + y^4 + z^5);\n"
		"(w != 0) and (x != 0) and (y != 0) and (z != 0);");
	return (expr);
	}



/****************************************************************************
 * howto_create_and_solve_an_expression
 ****************************************************************************/
static void
howto_create_and_solve_an_expression(
	int				print_solution)

	{
	struct isat3			*is3;
	struct isat3_node		*variable_w;
	struct isat3_node		*variable_x;
	struct isat3_node		*variable_y;
	struct isat3_node		*variable_z;
	struct isat3_node		*expr;
	i3_type_t			result;

	/* create an isat3 object */

	is3 = isat3_init(NULL);

	/*
	 * first declare variables occuring in the expression
	 * - use isat3_node_create_variable_boole() to declare a boolean
	 *   variable
	 * - use isat3_node_create_variable_integer() to declare an integer
	 *   variable
	 * - use isat3_node_create_variable_float() to declare a float
	 *   variable
	 * - every variable has to have a unique name
	 * - integer and float variables have to be declared with an
	 *   interval range
	 */

	variable_w = isat3_node_create_variable_integer(is3, "w", -100, 100);
	variable_x = isat3_node_create_variable_integer(is3, "x", -100, 100);
	variable_y = isat3_node_create_variable_integer(is3, "y", -100, 100);
	variable_z = isat3_node_create_variable_integer(is3, "z", -100, 100);

	/* build up ((w = x + 2*y + z) and (w < x)) using node operations */

	expr = howto_build_an_expression_with_node_operations(
		is3,
		variable_w,
		variable_x,
		variable_y,
		variable_z);

	/* solve the expression (with no timeout) */

	result = isat3_solve_expr(is3, expr, 0);

	/* print the result */

	if ((print_solution == 0) || (print_solution == 1))
		{
		printf("isat3_solve_expr() example1:\n");
		howto_examine_the_result(
			is3,
			variable_w,
			variable_x,
			variable_y,
			variable_z,
			NULL,
			NULL,
			result);
		}
	isat3_node_destroy(is3, expr);

	/*
	 * build up ((w^2 = x^3 + y^4 + z^5) and (w != 0) and (x != 0) and
	 * (y != 0) and (z != 0)) from a string
	 */

	expr = howto_build_an_expression_from_a_string(is3);

	/* solve the expression (with timeout 4.3 seconds = 4300000 us) */

	result = isat3_solve_expr(is3, expr, 4300000);

	/* print the result */

	if ((print_solution == 0) || (print_solution == 2))
		{
		printf("isat3_solve_expr() example2:\n");
		howto_examine_the_result(
			is3,
			variable_w,
			variable_x,
			variable_y,
			variable_z,
			NULL,
			NULL,
			result);
		isat3_node_destroy(is3, expr);
		}

	/* variables are not needed any more */

	isat3_node_destroy(is3, variable_z);
	isat3_node_destroy(is3, variable_y);
	isat3_node_destroy(is3, variable_x);
	isat3_node_destroy(is3, variable_w);

	/* destroy isat3 object */

	isat3_deinit(is3);
	}



/****************************************************************************
 * howto_create_and_solve_a_bmc_problem1
 ****************************************************************************/
static void
howto_create_and_solve_a_bmc_problem1(
	int				print_solution)

	{
	struct isat3			*is3;
	struct isat3_node		*variable_x;
	struct isat3_node		*variable_y;
	struct isat3_node		*init;
	struct isat3_node		*trans;
	struct isat3_node		*target;
	i3_type_t			result;

	/* create an isat3 object */

	is3 = isat3_init(NULL);

	/* first declare variables occuring in the problem */

	variable_x = isat3_node_create_variable_float(is3, "x", -100, 100);
	variable_y = isat3_node_create_variable_float(is3, "y", -100, 100);

	/* build up INIT, TRANS and TARGET from a string */

	init = isat3_node_create_from_string(is3,
		"y = 0;");
	trans = isat3_node_create_from_string(is3,
		"x >= y and x <= y + 6;\n"
		"sin(x) = 1;\n"
		"y' = y + 3.1415;");
	target = isat3_node_create_from_string(is3,
		"y >= 20;");

	/* solve the problem (with timeout 3.14 seconds = 3140000 us) */

	result = isat3_solve_bmc(is3, init, trans, target, 0, ISAT3_MAX_TFRAME, 3140000);

	/* print the result */

	if ((print_solution == 0) || (print_solution == 1))
		{
		printf("isat3_solve_bmc() example1:\n");
		howto_examine_the_result(
			is3,
			variable_x,
			variable_y,
			NULL,
			NULL,
			NULL,
			NULL,
			result);
		}
	isat3_node_destroy(is3, target);

	/* build up another TARGET */

	target = isat3_node_create_from_string(is3,
		"y >= 40;");

	/* solve the problem (without timeout) */

	result = isat3_solve_bmc(is3, init, trans, target, 0, ISAT3_MAX_TFRAME, 0);

	/* print the result */

	if ((print_solution == 0) || (print_solution == 2))
		{
		printf("isat3_solve_bmc() example2:\n");
		howto_examine_the_result(
			is3,
			variable_x,
			variable_y,
			NULL,
			NULL,
			NULL,
			NULL,
			result);
		}
	isat3_node_destroy(is3, target);
	isat3_node_destroy(is3, trans);
	isat3_node_destroy(is3, init);

	/* variables are not needed any more */

	isat3_node_destroy(is3, variable_y);
	isat3_node_destroy(is3, variable_x);

	/* destroy isat3 object */

	isat3_deinit(is3);
	}



/****************************************************************************
 * howto_create_and_solve_a_bmc_problem2
 ****************************************************************************/
static void
howto_create_and_solve_a_bmc_problem2(
	int				print_solution)

	{
	struct isat3			*is3;
	struct isat3_node		*variable_jump;
	struct isat3_node		*variable_n;
	struct isat3_node		*variable_t;
	struct isat3_node		*variable_xv;
	struct isat3_node		*variable_yv;
	struct isat3_node		*variable_xh;
	struct isat3_node		*variable_yh;
	struct isat3_node		*init;
	struct isat3_node		*init1;
	struct isat3_node		*trans;
	struct isat3_node		*target;
	struct isat3_node		*dummy;
	i3_type_t			result;

	/* create an isat3 object */

	is3 = isat3_init(NULL);

	/* first declare variables occuring in the problem */

	variable_jump = isat3_node_create_variable_boole(is3, "jump");
	variable_n = isat3_node_create_variable_integer(is3, "n", 0, 15);
	variable_t = isat3_node_create_variable_float(is3, "t", 0.0, 10000.0);
	variable_xv = isat3_node_create_variable_float(is3, "xv", -100.0, 100.0);
	variable_yv = isat3_node_create_variable_float(is3, "yv", -100.0, 100.0);
	variable_xh = isat3_node_create_variable_float(is3, "xh", -100.0, 100.0);
	variable_yh = isat3_node_create_variable_float(is3, "yh", -100.0, 100.0);

	/* build up INIT, TRANS and TARGET from string */

	init = isat3_node_create_from_string(is3,
		"xv <= 15.0;\n"
		"yv = 0.0;\n"
		"xh = 0.0;\n"
		"yh <= 5.0;\n"
		"n = 0;\n"
		"t = 0.0;\n");
	init1 = isat3_node_create_from_string(is3,
		"xv = 10.0;\n"
		"yv = 0.0;\n"
		"xh = 0.0;\n"
		"yh = 4.0;\n"
		"n = 0;\n"
		"t = 0.0;\n");
	trans = isat3_node_create_from_string(is3,
		"jump <-> (xv <= 0.0 and yv < 0.0);\n"
		"!jump -> (xv' = xv + 0.2 * yv  - 0.5 * 9.8 * 0.2^2 and\n"
		"         yv' = yv - 0.2 * 9.8 and\n"
		"         xh' = xh + 0.2 * yh and\n"
		"         yh' = yh and\n"
		"         n' = n and\n"
		"         t' = t + 0.2);\n"
		"jump ->  (xv' = 0.0 and\n"
		"         yv' = -(1.0 - 0.3) * yv and\n"
		"         xh' = xh and\n"
		"         yh' = (1.0 - 0.1) * yh and\n"
		"         n' = n + 1 and\n"
		"         t' = t);\n");
	target = isat3_node_create_from_string(is3,
		"xh = 40.0 and xv = 0.0;\n");
	dummy = isat3_node_create_constant_true(is3);

	/* solve the problem with INIT */

	result = isat3_solve_bmc(is3, init, trans, dummy, 10, 10, 0);
	if ((print_solution == 0) || (print_solution == 1))
		{
		printf("isat3_solve_bmc() bouncing ball with INIT:\n");
		howto_examine_the_result(
			is3,
			variable_n,
			variable_t,
			variable_xv,
			variable_yv,
			variable_xh,
			variable_yh,
			result);
		}

	/* solve the problem with INIT1 */

	result = isat3_solve_bmc(is3, init1, trans, dummy, 10, 10, 0);
	if ((print_solution == 0) || (print_solution == 2))
		{
		printf("isat3_solve_bmc() bouncing ball with INIT1:\n");
		howto_examine_the_result(
			is3,
			variable_n,
			variable_t,
			variable_xv,
			variable_yv,
			variable_xh,
			variable_yh,
			result);
		}

	isat3_node_destroy(is3, dummy);
	isat3_node_destroy(is3, target);
	isat3_node_destroy(is3, trans);
	isat3_node_destroy(is3, init1);
	isat3_node_destroy(is3, init);

	/* variables are not needed any more */

	isat3_node_destroy(is3, variable_yh);
	isat3_node_destroy(is3, variable_xh);
	isat3_node_destroy(is3, variable_yv);
	isat3_node_destroy(is3, variable_xv);
	isat3_node_destroy(is3, variable_t);
	isat3_node_destroy(is3, variable_n);
	isat3_node_destroy(is3, variable_jump);

	/* destroy isat3 object */

	isat3_deinit(is3);
	}



/****************************************************************************
 * howto_parse_strings_and_use_defines
 ****************************************************************************/
static void
howto_parse_strings_and_use_defines(
	void)

	{
	struct isat3			*is3;
	struct isat3_node		*constant;
	struct isat3_node		*define;
	struct isat3_node		*variable_x;
	struct isat3_node		*variable_y;
	struct isat3_node		*variable_z;
	struct isat3_node		*some_clauses;
	const i3_char_t			*clauses;

	/* create an isat3 object */

	is3 = isat3_init(NULL);

	/* create a constant and associate with a define */

	constant = isat3_node_create_constant_float(is3, 1000.5);
	define = isat3_node_create_define(is3, "magic_number", constant);
	isat3_node_destroy(is3, constant);

	/*
	 * this define may now be used in strings and will be replaced by
	 * associated constant
	 */

	variable_x = isat3_node_create_variable_float(is3, "x", 0, 1000);
	variable_y = isat3_node_create_variable_float(is3, "y", 0, 1000);
	variable_z = isat3_node_create_variable_float(is3, "z", 0, 1000);
	clauses =
		"magic_number * x + y < -z;\n"
		"z*z^2 = -y-y + magic_number + (z^3 + 4*y);\n";
	some_clauses = isat3_node_create_from_string(is3, clauses);

	/* ... */

	isat3_node_destroy(is3, some_clauses);
	isat3_node_destroy(is3, variable_x);
	isat3_node_destroy(is3, variable_y);
	isat3_node_destroy(is3, variable_z);
	isat3_node_destroy(is3, define);

	/* destroy isat3 object */

	isat3_deinit(is3);
	printf("parse strings done\n\n");
	}



/****************************************************************************
 * howto_use_ternary_nodes
 ****************************************************************************/
static void
howto_use_ternary_nodes(
	void)

	{
	struct isat3			*is3;
	struct isat3_node		*variable_a;
	struct isat3_node		*variable_b;
	struct isat3_node		*variable_c;
	struct isat3_node		*variable_d;
	struct isat3_node		*variable_e;
	struct isat3_node		*xor3;
	struct isat3_node		*nor4;
	struct isat3_node		*and5;

	/* create an isat3 object */

	is3 = isat3_init(NULL);
	variable_a = isat3_node_create_variable_boole(is3, "a");
	variable_b = isat3_node_create_variable_boole(is3, "b");
	variable_c = isat3_node_create_variable_boole(is3, "c");
	variable_d = isat3_node_create_variable_boole(is3, "d");
	variable_e = isat3_node_create_variable_boole(is3, "e");

	/* create some non-binary nodes */

	and5 = isat3_node_create_quinary_operation(
		is3,
		ISAT3_NODE_NOP_AND,
		variable_a,
		variable_b,
		variable_c,
		variable_d,
		variable_e);
	nor4 = isat3_node_create_quaternary_operation(
		is3,
		ISAT3_NODE_NOP_NOR,
		and5,
		variable_a,
		variable_b,
		variable_c);
	xor3 = isat3_node_create_ternary_operation(
		is3,
		ISAT3_NODE_NOP_XOR,
		nor4,
		variable_a,
		variable_b);

	/* ... */

	isat3_node_destroy(is3, xor3);
	isat3_node_destroy(is3, nor4);
	isat3_node_destroy(is3, and5);
	isat3_node_destroy(is3, variable_e);
	isat3_node_destroy(is3, variable_d);
	isat3_node_destroy(is3, variable_c);
	isat3_node_destroy(is3, variable_b);
	isat3_node_destroy(is3, variable_a);

	/* destroy isat3 object */

	isat3_deinit(is3);
	printf("ternary nodes done\n\n");
	}



/****************************************************************************
 * howto_use_nary_nodes
 ****************************************************************************/
static void
howto_use_nary_nodes(
	void)

	{
	struct isat3			*is3;
	struct isat3_node		*variables[5];
	struct isat3_node		*xor3;
	struct isat3_node		*nor4;
	struct isat3_node		*and5;

	/* create an isat3 object */

	is3 = isat3_init(NULL);
	variables[0] = isat3_node_create_variable_boole(is3, "a");
	variables[1] = isat3_node_create_variable_boole(is3, "b");
	variables[2] = isat3_node_create_variable_boole(is3, "c");
	variables[3] = isat3_node_create_variable_boole(is3, "d");
	variables[4] = isat3_node_create_variable_boole(is3, "e");

	/* create some non-binary nodes */

	and5 = isat3_node_create_nary_operation(
		is3,
		ISAT3_NODE_NOP_AND,
		variables,
		5);
	nor4 = isat3_node_create_quaternary_operation(
		is3,
		ISAT3_NODE_NOP_NOR,
		and5,
		variables[0],
		variables[1],
		variables[2]);
	xor3 = isat3_node_create_ternary_operation(
		is3,
		ISAT3_NODE_NOP_XOR,
		nor4,
		variables[0],
		variables[1]);

	/* ... */

	isat3_node_destroy(is3, xor3);
	isat3_node_destroy(is3, nor4);
	isat3_node_destroy(is3, and5);
	isat3_node_destroy(is3, variables[4]);
	isat3_node_destroy(is3, variables[3]);
	isat3_node_destroy(is3, variables[2]);
	isat3_node_destroy(is3, variables[1]);
	isat3_node_destroy(is3, variables[0]);

	/* destroy isat3 object */

	isat3_deinit(is3);
	printf("nary nodes done\n\n");
	}



/****************************************************************************
 * howto_do_incremental_solving
 ****************************************************************************/
static void
howto_do_incremental_solving(
	int				print_solution)

	{
	struct isat3			*is3;
	struct isat3_node		*variable_x;
	struct isat3_node		*variable_y;
	struct isat3_node		*variable_z;
	struct isat3_node		*constraints;
	i3_type_t			result;

	/* create an isat3 object */

	is3 = isat3_init(NULL);

	/* create variables */

	variable_x = isat3_node_create_variable_float(is3, "x", -1000, 1000);
	variable_y = isat3_node_create_variable_float(is3, "y", -1000, 1000);
	variable_z = isat3_node_create_variable_float(is3, "z", -1000, 1000);

	/* add a backtack point */

	isat3_push_btpoint(is3);

	/* create constraints */

	constraints = isat3_node_create_from_string(is3, "x + y < 7;\n");
	isat3_add_constraint(is3, constraints);
	isat3_node_destroy(is3, constraints);

	constraints = isat3_node_create_from_string(is3, "y + z > 10;\n");
	isat3_add_constraint(is3, constraints);
	isat3_node_destroy(is3, constraints);

	/* solve */

	result = isat3_solve_constraints(is3, 1000000);
	if ((print_solution == 0) || (print_solution == 1))
		{
		howto_examine_the_result(
			is3,
			variable_x,
			variable_y,
			variable_z,
			NULL,
			NULL,
			NULL,
			result);
		}

	/* add a backtack point */

	isat3_push_btpoint_name(is3, "this_is_the_second_backtrack_point");

	/* add constraints */

	constraints = isat3_node_create_from_string(is3, "y + z < 10;\n");
	isat3_add_constraint(is3, constraints);
	isat3_node_destroy(is3, constraints);

	/* solve */

	result = isat3_solve_constraints(is3, 1000000);
	if ((print_solution == 0) || (print_solution == 2))
		{
		howto_examine_the_result(
			is3,
			variable_x,
			variable_y,
			variable_z,
			NULL,
			NULL,
			NULL,
			result);
		}

	/* backtrack */

	isat3_pop_btpoint(is3, 2);

	/* solve */

	result = isat3_solve_constraints(is3, 1000000);
	if ((print_solution == 0) || (print_solution == 3))
		{
		howto_examine_the_result(
			is3,
			variable_x,
			variable_y,
			variable_z,
			NULL,
			NULL,
			NULL,
			result);
		}

	/* add constraints */

	constraints = isat3_node_create_from_string(
		is3,
		"y + z < -10;\n"
		"y + z > -20;\n"
		"y = 100;\n");
	isat3_add_constraint(is3, constraints);
	isat3_node_destroy(is3, constraints);

	/* solve */

	result = isat3_solve_constraints(is3, 1000000);
	if ((print_solution == 0) || (print_solution == 4))
		{
		howto_examine_the_result(
			is3,
			variable_x,
			variable_y,
			variable_z,
			NULL,
			NULL,
			NULL,
			result);
		}

	/* destroy the variables */

	isat3_node_destroy(is3, variable_z);
	isat3_node_destroy(is3, variable_y);
	isat3_node_destroy(is3, variable_x);

	/* destroy isat3 object */

	isat3_deinit(is3);
	}



/****************************************************************************
 * howto_set_a_trace_file
 ****************************************************************************/
static void
howto_set_a_trace_file(
	void)

	{
	struct isat3_settings		*is3_set;
	struct isat3			*is3;
	struct isat3_node		*variables[4];
	struct isat3_node		*add;
	struct isat3_node		*expr;
	const i3_char_t			path[] = "libisat3_example.trace";

	/*
	 * create an isat3 object and set the file to which every API
	 * call will be logged
	 */

	printf("writing to '%s'\n\n", path);
	is3_set = isat3_settings_init(NULL);
	isat3_settings_set_trace_file(is3_set, path);
	is3 = isat3_init_with_settings(NULL, is3_set);
	isat3_settings_deinit(is3_set);

	/* declare variables */

	variables[0] = isat3_node_create_variable_integer(is3, "w", -100, 100);
	variables[1]= isat3_node_create_variable_integer(is3, "x", -100, 100);
	variables[2] = isat3_node_create_variable_integer(is3, "y", -100, 100);
	variables[3] = isat3_node_create_variable_integer(is3, "z", -100, 100);

	/* dummy add */

	add = isat3_node_create_nary_operation(
		is3,
		ISAT3_NODE_NOP_ADD,
		variables,
		4);
	isat3_node_destroy(is3, add);

	/*
	 * build up ((w^2 = x^3 + y^4 + z^5) and (w != 0) and (x != 0) and
	 * (y != 0) and (z != 0)) from a string
	 */

	expr = howto_build_an_expression_from_a_string(is3);

	/* solve the expression (with timeout 4.3 seconds = 4300000 us) */

	isat3_solve_expr(is3, expr, 4300000);
	isat3_node_destroy(is3, expr);

	/* variables are not needed any more */

	isat3_node_destroy(is3, variables[3]);
	isat3_node_destroy(is3, variables[2]);
	isat3_node_destroy(is3, variables[1]);
	isat3_node_destroy(is3, variables[0]);

	/* destroy isat3 object */

	isat3_deinit(is3);
	}



/****************************************************************************
 * argument_is
 ****************************************************************************/
static int
argument_is(
	int				argc,
	char				**argv,
	const char			*wanted)

	{
	if (argc <= 1) return (0);
	return (strcmp(argv[1], wanted) == 0);
	}



/****************************************************************************
 * no_argument_or_argument_is
 ****************************************************************************/
static int
no_argument_or_argument_is(
	int				argc,
	char				**argv,
	const char			*wanted)

	{
	if (argc <= 1) return (1);
	return (argument_is(argc, argv, wanted));
	}



/****************************************************************************
 * main
 ****************************************************************************/
int
main(
	int				argc,
	char				**argv)

	{

	/* setup once at program start */

	isat3_setup();

	/*
	 * this example shows how to:
	 *
	 * 1. solve an expression (EXPR is an isat3_node)
	 *    - use functions isat3_node_create_*() to build up EXPR
	 *    - call isat3_solve_expr()
	 *    - currently there is no sharing of learned knowledge between
	 *      subsequent calls
	 *
	 * 2. solve a bmc problem (INIT, TRANS and TARGET are isat3_nodes)
	 *    - use functions isat3_node_create_*() to build up INIT,
	 *      TRANS and TARGET
	 *    - call isat3_solve_bmc()
	 *    - currently there is no sharing of learned knowledge between
	 *      subsequent calls
	 *
	 * 3. solve incrementelly
	 *    - use functions isat3_node_create_*() to build up constraints
	 *    - add them with isat3_add_constraint()
	 *    - call isat3_solve_constraints() to solve current conjunction
	 *      of constraints
	 *    - use isat3_push_btpoint() and isat3_pop_btpoint() to create
	 *      backtrack points, to remove added constraints
	 *    - there is sharing of learned knowledge between subsequent
	 *      calls to isat3_solve_constraints()
	 *
	 * for debugging purposes it is possible to log every API-call. this
	 * can be enabled with a special variant of isat3_init()
	 */

	/* usage examples */

	if (no_argument_or_argument_is(argc, argv, "expression"))
		{
		howto_create_and_solve_an_expression(0);
		}
	if (argument_is(argc, argv, "expression_1"))
		{
		howto_create_and_solve_an_expression(1);
		}
	if (argument_is(argc, argv, "expression_2"))
		{
		howto_create_and_solve_an_expression(2);
		}

	if (no_argument_or_argument_is(argc, argv, "bmc_problem1"))
		{
		howto_create_and_solve_a_bmc_problem1(0);
		}
	if (argument_is(argc, argv, "bmc_problem1_1"))
		{
		howto_create_and_solve_a_bmc_problem1(1);
		}
	if (argument_is(argc, argv, "bmc_problem1_2"))
		{
		howto_create_and_solve_a_bmc_problem1(2);
		}

	if (no_argument_or_argument_is(argc, argv, "bmc_problem2"))
		{
		howto_create_and_solve_a_bmc_problem2(0);
		}
	if (argument_is(argc, argv, "bmc_problem2_1"))
		{
		howto_create_and_solve_a_bmc_problem2(1);
		}
	if (argument_is(argc, argv, "bmc_problem2_2"))
		{
		howto_create_and_solve_a_bmc_problem2(2);
		}

	if (no_argument_or_argument_is(argc, argv, "parse_strings_and_use_defines"))
		{
		howto_parse_strings_and_use_defines();
		}

	if (no_argument_or_argument_is(argc, argv, "use_ternary_nodes"))
		{
		howto_use_ternary_nodes();
		}

	if (no_argument_or_argument_is(argc, argv, "use_nary_nodes"))
		{
		howto_use_nary_nodes();
		}

	if (no_argument_or_argument_is(argc, argv, "do_incremental_solving"))
		{
		howto_do_incremental_solving(0);
		}
	if (argument_is(argc, argv, "do_incremental_solving_1"))
		{
		howto_do_incremental_solving(1);
		}
	if (argument_is(argc, argv, "do_incremental_solving_2"))
		{
		howto_do_incremental_solving(2);
		}
	if (argument_is(argc, argv, "do_incremental_solving_3"))
		{
		howto_do_incremental_solving(3);
		}
	if (argument_is(argc, argv, "do_incremental_solving_4"))
		{
		howto_do_incremental_solving(4);
		}

	if (no_argument_or_argument_is(argc, argv, "trace_file"))
		{
		howto_set_a_trace_file();
		}

	/* cleanup once at program end */

	isat3_cleanup();
	return (0);
	}



/****************************************************************************/
